﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;

using Advent.aiDevices;

namespace Advent.Controls
{
    public partial class AI5620TelInt : UserControl
    {

        AI5620_TE_Simulator.TelephoneInterface TelInt;  // Capabilities
        AI5620Capabilities Capabilities;                // AI-5620 capabilities


        public AI5620TelInt()
        {
            InitializeComponent();

            // Load the available termination bandwidth settings
            cbo_ACTermBandwidth.Items.Clear();
            foreach (AI5620_TE_Simulator.TerminationBandwidth BW
                      in Enum.GetValues(typeof(AI5620_TE_Simulator.TerminationBandwidth)))
            {
                cbo_ACTermBandwidth.Items.Add(BW);
            }
            cbo_ACTermBandwidth.SelectedIndex = 0;

            //Load the available termination modes
            cbo_Terminationmode.Items.Clear();
            foreach (AI5620_TE_Simulator.ACTerminationSetting S
                in Enum.GetValues(typeof(AI5620_TE_Simulator.ACTerminationSetting)))
            {
                cbo_Terminationmode.Items.Add(S);
            }
            cbo_Terminationmode.SelectedIndex = 0;


            //Load the available fixed terminations
            cbo_FixedTermination.Items.Clear();
            foreach (AI5620_TE_Simulator.FixedTerminationValue Z
                in Enum.GetValues(typeof(AI5620_TE_Simulator.FixedTerminationValue)))
            {
                cbo_FixedTermination.Items.Add(Z);
            }
            cbo_FixedTermination.SelectedIndex = 0;


            // Create all the neccessary event handlers

            rb_ComplexAC.Click += new EventHandler(UpdateACImpedance);
            rb_ResistiveAC.Click += new EventHandler(UpdateACImpedance);
            cbo_ACImpedance.SelectedIndexChanged += new EventHandler(UpdateACImpedance);
            txt_ACResistance.Validating += new CancelEventHandler(txt_ACResistance_Validating);
            txt_ACResistance.KeyUp += new KeyEventHandler(txt_ACResistance_KeyUp);


            rb_MaxOnHookResistance.Click += new EventHandler(UpdateOnHookDCResistance);
            rb_OnHookR.Click += new EventHandler(UpdateOnHookDCResistance);
            txt_OnHookDCR.Validating += new CancelEventHandler(txt_OnHookDCR_Validating);
            txt_OnHookDCR.KeyUp += new KeyEventHandler(txt_OnHookDCR_KeyUp);

            txt_OffHookResistance.Validating += new CancelEventHandler(txt_OffHookResistance_Validating);
            txt_OffHookResistance.KeyUp += new KeyEventHandler(txt_OffHookResistance_KeyUp);
        }






        /// <summary>
        /// This assigns the AI-5620 telephone interface which will interact with this control
        /// </summary>
        public void SetTelInt(AI5620_TE_Simulator.TelephoneInterface TelInt,AI5620Capabilities Capabilities)
        {
            this.TelInt = TelInt;
            this.Capabilities = Capabilities;

            if (TelInt == null) return;
            
            // Update the list of fixed impedances available
            cbo_ACImpedance.Items.Clear();
            foreach(Impedance Z in TelInt.FixedImpedancesAvailable)
            {
                cbo_ACImpedance.Items.Add(Z);
            }
            cbo_ACImpedance.SelectedIndex = 0;

            cbo_REN.Items.Clear();
            cbo_REN.Items.Add((double)0);
            foreach (double REN in Capabilities.AvailableRingingLoads)
            {
                cbo_REN.Items.Add(REN);
            }
            cbo_REN.SelectedIndex = 0;



            RefreshControls();
        }

        /// <summary>
        /// This refreshes the contents of the controls from the telephone interface
        /// </summary>
        private void RefreshControls()
        {
            try
            {
                Impedance Z = TelInt.ACImpedance;

                ResistiveImpedance R = Z as ResistiveImpedance;
                if (R != null)
                {
                    rb_ResistiveAC.Checked = (R!=null);
                    if (R!=null)
                    {
                        txt_ACResistance.Text = R.Resistance.Value.ToString();
                    }
                }
                else
                {
                   rb_ComplexAC.Checked=true;
                   cbo_ACImpedance.SelectedItem = Z;
                }

                cbo_ACTermBandwidth.SelectedItem = TelInt.ACTerminationBandwidth;
                cbo_Terminationmode.SelectedItem = TelInt.ACTerminationMode;
                cbo_Balance.SelectedItem = TelInt.Balance;
                cbo_REN.SelectedItem = TelInt.RingerLoad;
                chk_HighGainMuteTX.Checked = TelInt.HighGainMuteTx;
                txt_OffHookResistance.Text = TelInt.OffHookDCResistance.Value.ToString();

                Resistance OnHookR = TelInt.OnHookDCResistance;
                if (OnHookR == Capabilities.OnHookDCResistanceMaximum)
                {
                    rb_MaxOnHookResistance.Checked = true;

                }
                else
                {
                    rb_OnHookR.Checked = true;
                    txt_OnHookDCR.Text = OnHookR.ValueInKilohms.ToString();
                }
                cbo_FixedTermination.SelectedItem = TelInt.FixedTermination;
                chk_FastSettle.Checked = TelInt.FastSettle;

            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }

        }


        #region EventHandlers

        void txt_ACResistance_KeyUp(object sender, KeyEventArgs e)
        {
            if (e.KeyCode != Keys.Return) return;
            UpdateACImpedance(null, EventArgs.Empty);
        }

        void txt_ACResistance_Validating(object sender, CancelEventArgs e)
        {
            UpdateACImpedance(null, EventArgs.Empty);
        }

        /// <summary>
        /// This handles all events which relate to updating the AC impedance settings
        /// </summary>
        void UpdateACImpedance(object sender, EventArgs e)
        {
            try
            {
                if (TelInt == null) return;
                if (rb_ComplexAC.Checked)
                {
                    TelInt.ACImpedance = (Impedance)cbo_ACImpedance.SelectedItem;
                }
                else
                {
                    double R;
                    if (!double.TryParse(txt_ACResistance.Text, out R)) throw new Exception("Invalid AC Resistance");
                    TelInt.ACImpedance = ResistiveImpedance.InOhms(R);



                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                RefreshControls();
            }
        }
        /// <summary>
        /// This updates the AC termination bandwidth
        /// </summary>
        private void cbo_ACTermBandwidth_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (TelInt == null) return;
            TelInt.ACTerminationBandwidth = (AI5620_TE_Simulator.TerminationBandwidth) cbo_ACTermBandwidth.SelectedItem;
        }
        /// <summary>
        /// This updates the termination mode
        /// </summary>
        private void cbo_Terminationmode_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (TelInt == null) return;
            TelInt.ACTerminationMode = (AI5620_TE_Simulator.ACTerminationSetting) cbo_Terminationmode.SelectedItem;

        }

        #region Ringing Load
        private void cbo_REN_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (TelInt == null) return;
            TelInt.RingerLoad = (double)cbo_REN.SelectedItem;
        }
        #endregion


        #region Off Hook DC resistance

        void txt_OffHookResistance_KeyUp(object sender, KeyEventArgs e)
        {
            if (e.KeyCode != Keys.Return) return;
            UpdateOffHookResistance();
        }

        void txt_OffHookResistance_Validating(object sender, CancelEventArgs e)
        {
            UpdateOffHookResistance();
        }

        private void UpdateOffHookResistance()
        {
            try
            {
                double R;
                if (!double.TryParse(txt_OffHookResistance.Text, out R)) throw new Exception("Invalid Off Hook DC Resistance");
                TelInt.OffHookDCResistance = Resistance.InOhms(R);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                RefreshControls();
            }

        }


        #endregion



        #region On Hook DC Resistance


        void txt_OnHookDCR_KeyUp(object sender, KeyEventArgs e)
        {
            if (e.KeyCode != Keys.Return) return;
            UpdateOnHookDCResistance(null, EventArgs.Empty);
        }
        void txt_OnHookDCR_Validating(object sender, CancelEventArgs e)
        {
            UpdateOnHookDCResistance(null, EventArgs.Empty);
        }


        void UpdateOnHookDCResistance(object sender, EventArgs e)
        {
            try
            {
                double R;
                if (!double.TryParse(txt_OnHookDCR.Text, out R)) throw new Exception("Invalid On Hook DC Resistance");


                if (rb_MaxOnHookResistance.Checked)
                {
                    TelInt.OnHookDCResistance = Capabilities.OnHookDCResistanceMaximum;
                }
                else
                {
                    TelInt.OnHookDCResistance = Resistance.InKilohms(R);
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                RefreshControls();
            }
        }

        #endregion




        private void chk_HighGainMuteTX_CheckedChanged(object sender, EventArgs e)
        {
            if (TelInt == null) return;
            TelInt.HighGainMuteTx = chk_HighGainMuteTX.Checked;
        }

        private void cbo_FixedTermination_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (TelInt == null) return;
            TelInt.FixedTermination = (AI5620_TE_Simulator.FixedTerminationValue)cbo_FixedTermination.SelectedItem;
        }
        private void chk_FastSettle_CheckedChanged(object sender, EventArgs e)
        {
            if (TelInt == null) return;
            TelInt.FastSettle = chk_FastSettle.Checked;
        }
        #endregion




        #region Actions




        private void bnt_Connect_Click(object sender, EventArgs e)
        {
            try
            {
                TelInt.Connect();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void btn_Disconnect_Click(object sender, EventArgs e)
        {
            try
            {
                TelInt.Disconnect();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void btn_GoOffHook_Click(object sender, EventArgs e)
        {
            try
            {
                TelInt.GoOffHook();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void btn_GoOnHook_Click(object sender, EventArgs e)
        {
            try
            {
                TelInt.GoOnHook();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void btn_Flash_Click(object sender, EventArgs e)
        {
            try
            {
                TelInt.Generate(new LineFlash(TimeInterval.InMilliseconds(200)));
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void lbl_ResetToDefaults_Click(object sender, EventArgs e)
        {
            try
            {
                TelInt.ResetToDefaults();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
            finally
            {
                RefreshControls();
            }
        }
        #endregion
















    }
}
